% weight matrices of the attractors
% eric zilli - 20111117 - v1.0
%
% This script prequires that the necessary weight matrices generated by 
% FuhsTouretzky2006.m, GuanellaEtAl2007.m, and BurakFiete2009.m have all
% saved to disk to make this figure. See the commented code after the
% opening comments in those scripts for the code to save the variables
% needed by this script.
%

% arrangement:
% columns:             single cell    one bump    full grid    moving N
% rows:
% Fu06 activity
% Fu06 net syn. input
% Gu07 activity
% Gu07 net syn. input
% Bu09 activity
% Bu09 net syn. input

%% Load variables
load data/Fu06_WeightFigure_vars.mat;
load data/Gu07_WeightFigure_vars.mat;
load data/Bu09_WeightFigure_vars.mat;

plotVars = {'Bu09_n1_netsyn','Bu09_bump_netsyn','Bu09_full_netsyn','Bu09_north_netsyn',...
            'Bu09_n1_act','Bu09_bump_act','Bu09_full_act','Bu09_north_act',...
            'Gu07_n1_netsyn','Gu07_bump_netsyn','Gu07_full_netsyn','Gu07_north_netsyn',...
            'Gu07_n1_act','Gu07_bump_act','Gu07_full_act','Gu07_north_act',...
            'Fu06_n1_netsyn','Fu06_bump_netsyn','Fu06_full_netsyn','Fu06_north_netsyn'...
            'Fu06_n1_act','Fu06_bump_act','Fu06_full_act','Fu06_north_act'};

modelNames = {{'Burak and','Fiete 2009'},...
              {'Guanella','et al. 2007'},...
              {'Fuhs and','Touretzky 2006'}};

columnLabels = {{'Single cell','active'},...
                {'Single bump','active'},...
                {'Full pattern','active'},...
                {'Moving','north'}};

panelLabels = {'(A)','(B)','(C)'};

%% Make the colormap

len = 256; % resolution of color map

keyPoints = linspace(0,1,7);

redPoints =   [0   0.3  0.55 1 202/255 166/255 0.53];
greenPoints = [0   0.45 0.62 1 132/255 94/255 0];
bluePoints =  [0.3 0.6  0.7  1 40/255 0/255 0];


BCWYR2 = [interp1(keyPoints,redPoints,linspace(0,1,len),'linear','extrap'); ...
          interp1(keyPoints,greenPoints,linspace(0,1,len),'linear','extrap'); ...
          interp1(keyPoints,bluePoints,linspace(0,1,len),'linear','extrap')]';

% Shift+scale matrix so its zero is at 0.5 (so we can multiply it by the
% length of the colormap to get the 0 point of the input matrix at the 0
% point in the colormap, which is halfway through its length). The first
% function actually plots the matrix, the second just returns it.
imagepn = @(v)image(len*(v./2/abs(eps+max(abs([max(max(v)) min(min(v))])))+0.5));
imagepnc = @(v)(len*(v./2/abs(eps+max(abs([max(max(v)) min(min(v))])))+0.5));


%% Figure options:
set(0,'defaultAxesFontName', 'Arial')
set(0,'defaultTextFontName', 'Arial')

% size on paper:
widthOnPaper = 18; % cm
heightOnPaper = 18; % cm

figure('units','centimeters','position',[1 1 widthOnPaper heightOnPaper],'color','w');
set(gcf, 'renderer', 'painter')
set(gcf, 'PaperUnits', 'centimeters');
set(gcf, 'PaperSize', [widthOnPaper heightOnPaper]);
set(gcf, 'PaperPositionMode', 'manual');
set(gcf, 'PaperPosition', [0 0 widthOnPaper heightOnPaper]);
colormap(BCWYR2);

% Variables for positioning plots
nRows = 6;
nCols = 5;

leftMargin = 0.03;
bottomMargin = 0.005;
lefts = leftMargin + 0.9*(0:nCols-1)/nCols;
bottoms = bottomMargin + 0.93*(0:nRows-1)/nRows;
width = 0.9/nCols;
height = 0.9/nRows;

for r=1:nRows
  % First column is row labels:
  axes('position',[0 bottoms(r) width height]);
  axis off
  xlim([0 1])
  ylim([0 1])
  
  % Label rows
  if mod(r,2)==0
    text(1.1,.5,{'Neural','activities'},'horizontalalignment','right','fontsize',8)
    text(0.5,0,modelNames{r/2},'horizontalalignment','center','verticalalignment','middle','fontsize',11)
  else
    text(1.1,.5,{'Net synaptic','input'},'horizontalalignment','right','fontsize',8)
  end
  
  % Every other row has a panel label
  if mod(r,2)==1
    text(0.1, 2,...
      panelLabels{4-(r+1)/2},...
      'FontSize',10,...
      'FontWeight','bold',...
      'HorizontalAlignment','center');
  end
  
  % Remaining columns are data:
  for c=1:(nCols-1)
    ind = sub2ind([nCols-1 nRows],c,r);
    axes('position',[width+lefts(c) bottoms(r) width*0.9 height]);
    % I know, eval is not the greatest way to do this:
    eval(sprintf('imagepn(%s);',plotVars{ind}))
    
    axis equal
    set(gca,'box','on','xtick',[],'ytick',[])

    if r==nRows
      % Draw column labels in first row
      text(32,-16,columnLabels{c},'horizontalalignment','center','fontsize',11)
    end
    if r==nRows && c==(nCols-1)
      % Draw colorbar
      axes('position',[width+lefts(c)+width bottoms(r) width*0.1 height]);
      imagesc((1:length(BCWYR2))')
      set(gca,'ydir','normal')
      axis off
      text(1.75,0,'-','fontsize',9,'verticalalignment','middle')
      text(1.75,0.5*length(BCWYR2),'0','fontsize',9,'verticalalignment','middle')
      text(1.75,length(BCWYR2),'+','fontsize',9,'verticalalignment','middle')
    end
  end
end